@extends('layouts.app')

@section('title', 'Book Sales - The Smart School Education')
@section('page-title', 'Book Sales')

@section('content')
@if($bookItems->count() == 0)
<div class="bg-blue-50 border-l-4 border-blue-500 p-4 mb-6 rounded-lg">
    <div class="flex">
        <div class="flex-shrink-0">
            <svg class="h-5 w-5 text-blue-400" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
            </svg>
        </div>
        <div class="ml-3 flex-1">
            <p class="text-sm text-blue-700">
                <strong>No books in stock!</strong> To sell books, you need to add them to stock first. 
                <a href="{{ route('stock.create') }}" class="underline font-medium">Click here to add books to stock</a>
            </p>
        </div>
    </div>
</div>
@endif

<!-- Success Message -->
@if(session('success'))
<div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg">
    <div class="flex">
        <div class="flex-shrink-0">
            <svg class="h-5 w-5 text-green-400" fill="currentColor" viewBox="0 0 20 20">
                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
            </svg>
        </div>
        <div class="ml-3">
            <p class="text-sm text-green-700">{{ session('success') }}</p>
        </div>
    </div>
</div>
@endif

<!-- Header with Button -->
<div class="mb-6">
    <div class="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-4">
        <div>
            <h2 class="text-xl font-semibold text-gray-800">Book Sales</h2>
            <p class="text-sm text-gray-600">Manage book sales to students</p>
        </div>
        <div class="flex flex-col sm:flex-row gap-2 sm:gap-3 w-full sm:w-auto">
            <a href="{{ route('stock.create') }}" class="text-sm text-blue-600 hover:text-blue-800 flex items-center justify-center sm:justify-start px-4 py-2 border border-blue-600 rounded-lg w-full sm:w-auto">
                <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                </svg>
                <span class="text-sm sm:text-base">Add Book to Stock</span>
            </a>
            @if($bookItems->count() > 0)
            <button onclick="openModal()" class="bg-green-600 hover:bg-green-700 text-white px-4 sm:px-6 py-2 rounded-lg flex items-center justify-center sm:justify-start w-full sm:w-auto">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4v16m8-8H4"></path>
                </svg>
                <span class="text-sm sm:text-base">Sell Book</span>
            </button>
            @endif
        </div>
    </div>
</div>

<!-- Book Sales Table -->
<div class="bg-white rounded-lg shadow overflow-hidden mb-6">
    <div class="px-4 sm:px-6 py-4 border-b border-gray-200">
        <h3 class="text-base sm:text-lg font-semibold text-gray-900">Recent Book Sales</h3>
    </div>
    <div class="overflow-x-auto">
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Student</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Book</th>
                    <th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase">Quantity</th>
                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Unit Price</th>
                    <th class="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase">Total Amount</th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                @forelse($recentSales as $sale)
                <tr class="hover:bg-gray-50">
                    <td class="px-4 py-3 text-sm text-gray-900">{{ $sale->purchase_date->format('d M Y') }}</td>
                    <td class="px-4 py-3 text-sm text-gray-900">
                        {{ $sale->student->admission_number }} - {{ $sale->student->name }}
                    </td>
                    <td class="px-4 py-3 text-sm text-gray-600">{{ $sale->stockItem->name }}</td>
                    <td class="px-4 py-3 text-sm text-gray-500">{{ $sale->quantity }}</td>
                    <td class="px-4 py-3 text-sm text-gray-500 text-right">Rs. {{ number_format($sale->unit_price, 2) }}</td>
                    <td class="px-4 py-3 text-sm font-medium text-green-600 text-right">Rs. {{ number_format($sale->total_amount, 2) }}</td>
                </tr>
                @empty
                <tr>
                    <td colspan="6" class="px-6 py-4 text-center text-sm text-gray-500">
                        No book sales recorded yet. Click "Sell Book" to record your first sale.
                    </td>
                </tr>
                @endforelse
            </tbody>
        </table>
    </div>
</div>

<!-- Modal for Book Sale Form -->
<div id="bookSaleModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 hidden overflow-y-auto h-full w-full z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-3/4 lg:w-1/2 shadow-lg rounded-md bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-xl font-semibold text-gray-900">Sell Book to Student</h3>
            <button onclick="closeModal()" class="text-gray-400 hover:text-gray-600">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                </svg>
            </button>
        </div>

        <form action="{{ route('sales.store-book') }}" method="POST" id="bookSaleForm">
            @csrf

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <!-- Student -->
                <div>
                    <label for="modal_student_id" class="block text-sm font-medium text-gray-700 mb-2">Select Student *</label>
                    <select name="student_id" id="modal_student_id" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                        <option value="">Select Student</option>
                        @foreach($students as $student)
                        <option value="{{ $student->id }}">
                            {{ $student->admission_number }} - {{ $student->name }}
                        </option>
                        @endforeach
                    </select>
                </div>

                <!-- Book Item -->
                <div>
                    <label for="modal_stock_item_id" class="block text-sm font-medium text-gray-700 mb-2">Select Book *</label>
                    <select name="stock_item_id" id="modal_stock_item_id" required
                            class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                            onchange="updateModalPrice()">
                        <option value="">Select Book</option>
                        @foreach($bookItems as $item)
                        <option value="{{ $item->id }}" 
                                data-price="{{ $item->unit_price }}"
                                data-stock="{{ $item->quantity }}">
                            {{ $item->name }} (Stock: {{ $item->quantity }}) - Rs. {{ number_format($item->unit_price, 2) }}
                        </option>
                        @endforeach
                    </select>
                </div>

                <!-- Quantity -->
                <div>
                    <label for="modal_quantity" class="block text-sm font-medium text-gray-700 mb-2">Quantity *</label>
                    <input type="number" name="quantity" id="modal_quantity" value="1" min="1" required
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                           onchange="calculateModalTotal()">
                    <p class="mt-1 text-xs text-gray-500" id="modal-stock-info"></p>
                </div>

                <!-- Unit Price -->
                <div>
                    <label for="modal_unit_price" class="block text-sm font-medium text-gray-700 mb-2">Unit Price (Rs.) *</label>
                    <input type="number" name="unit_price" id="modal_unit_price" value="0" step="0.01" min="0" required
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"
                           onchange="calculateModalTotal()">
                </div>

                <!-- Purchase Date -->
                <div>
                    <label for="modal_purchase_date" class="block text-sm font-medium text-gray-700 mb-2">Purchase Date *</label>
                    <input type="date" name="purchase_date" id="modal_purchase_date" value="{{ date('Y-m-d') }}" required
                           class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500">
                </div>

                <!-- Total Amount (Read-only) -->
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">Total Amount</label>
                    <div class="w-full px-4 py-2 bg-gray-100 border border-gray-300 rounded-lg text-lg font-bold text-gray-900" id="modal_total_amount">
                        Rs. 0.00
                    </div>
                </div>
            </div>

            <!-- Notes -->
            <div class="mt-4">
                <label for="modal_notes" class="block text-sm font-medium text-gray-700 mb-2">Notes</label>
                <textarea name="notes" id="modal_notes" rows="2"
                          class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500"></textarea>
            </div>

            <!-- Buttons -->
            <div class="mt-6 flex justify-end gap-4">
                <button type="button" onclick="closeModal()" class="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50">
                    Cancel
                </button>
                <button type="submit" class="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700">
                    Record Sale & Update Ledger
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function openModal() {
    document.getElementById('bookSaleModal').classList.remove('hidden');
    document.body.style.overflow = 'hidden';
}

function closeModal() {
    document.getElementById('bookSaleModal').classList.add('hidden');
    document.body.style.overflow = 'auto';
    // Reset form
    document.getElementById('bookSaleForm').reset();
    document.getElementById('modal_total_amount').textContent = 'Rs. 0.00';
    document.getElementById('modal-stock-info').textContent = '';
    document.getElementById('modal_purchase_date').value = '{{ date('Y-m-d') }}';
}

function updateModalPrice() {
    const select = document.getElementById('modal_stock_item_id');
    const selectedOption = select.options[select.selectedIndex];
    const price = selectedOption.getAttribute('data-price');
    const stock = selectedOption.getAttribute('data-stock');
    
    if (price) {
        document.getElementById('modal_unit_price').value = price;
        document.getElementById('modal-stock-info').textContent = `Available Stock: ${stock} units`;
    }
    calculateModalTotal();
}

function calculateModalTotal() {
    const quantity = parseFloat(document.getElementById('modal_quantity').value) || 0;
    const unitPrice = parseFloat(document.getElementById('modal_unit_price').value) || 0;
    const total = quantity * unitPrice;
    document.getElementById('modal_total_amount').textContent = 'Rs. ' + total.toFixed(2);
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('bookSaleModal');
    if (event.target == modal) {
        closeModal();
    }
}

// Close modal on Escape key
document.addEventListener('keydown', function(event) {
    if (event.key === 'Escape') {
        closeModal();
    }
});
</script>
@endsection
