<?php

namespace App\Http\Controllers;

use App\Models\Student;
use Illuminate\Http\Request;

class StudentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Student::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('admission_number', 'like', "%{$search}%")
                  ->orWhere('father_name', 'like', "%{$search}%")
                  ->orWhere('class', 'like', "%{$search}%");
            });
        }

        $students = $query->latest()->paginate(15);

        return view('students.index', compact('students'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('students.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'admission_number' => 'required|unique:students,admission_number',
            'name' => 'required|string|max:255',
            'father_name' => 'nullable|string|max:255',
            'admission_date' => 'required|date',
            'class' => 'nullable|string|max:50',
            'admission_form' => 'nullable|numeric|min:0',
            'monthly_fees' => 'nullable|numeric|min:0',
            'annual_charges' => 'nullable|numeric|min:0',
            'remarks' => 'nullable|string',
        ]);

        Student::create($validated);

        return redirect()->route('students.index')
            ->with('success', 'Student registered successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Student $student)
    {
        $student->load([
            'feePayments.feeType', 
            'bookPurchases.stockItem', 
            'uniformPurchases.stockItem'
        ]);
        
        return view('students.show', compact('student'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Student $student)
    {
        return view('students.edit', compact('student'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Student $student)
    {
        $validated = $request->validate([
            'admission_number' => 'required|unique:students,admission_number,' . $student->id,
            'name' => 'required|string|max:255',
            'father_name' => 'nullable|string|max:255',
            'admission_date' => 'required|date',
            'class' => 'nullable|string|max:50',
            'admission_form' => 'nullable|numeric|min:0',
            'monthly_fees' => 'nullable|numeric|min:0',
            'annual_charges' => 'nullable|numeric|min:0',
            'remarks' => 'nullable|string',
        ]);

        $student->update($validated);

        return redirect()->route('students.index')
            ->with('success', 'Student updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Student $student)
    {
        $student->delete();

        return redirect()->route('students.index')
            ->with('success', 'Student deleted successfully!');
    }

    /**
     * Print students list
     */
    public function print(Request $request)
    {
        $query = Student::query();

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('admission_number', 'like', "%{$search}%")
                  ->orWhere('father_name', 'like', "%{$search}%")
                  ->orWhere('class', 'like', "%{$search}%");
            });
        }

        $students = $query->latest()->get();

        return view('students.print', compact('students'));
    }

    /**
     * Print single student details
     */
    public function printSingle(Student $student)
    {
        $student->load(['feePayments.feeType', 'bookPurchases.stockItem', 'uniformPurchases.stockItem']);
        return view('students.print-single', compact('student'));
    }
}
