<?php

namespace App\Http\Controllers;

use App\Models\StockItem;
use Illuminate\Http\Request;

class StockController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = StockItem::query();

        // Filter by type
        if ($request->has('type') && $request->type) {
            $query->where('type', $request->type);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        $stockItems = $query->latest()->paginate(15);
        $lowStockCount = StockItem::whereRaw('quantity <= alert_threshold')->count();

        return view('stock.index', compact('stockItems', 'lowStockCount'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('stock.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:book,uniform',
            'quantity' => 'required|integer|min:0',
            'alert_threshold' => 'required|integer|min:0',
            'unit_price' => 'required|numeric|min:0',
            'arrival_date' => 'nullable|date',
            'description' => 'nullable|string',
        ]);

        StockItem::create($validated);

        return redirect()->route('stock.index')
            ->with('success', 'Stock item added successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $stockItem = StockItem::with(['bookPurchases.student', 'uniformPurchases.student'])->findOrFail($id);
        return view('stock.show', compact('stockItem'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $stockItem = StockItem::findOrFail($id);
        return view('stock.edit', compact('stockItem'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $stockItem = StockItem::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:book,uniform',
            'quantity' => 'required|integer|min:0',
            'alert_threshold' => 'required|integer|min:0',
            'unit_price' => 'required|numeric|min:0',
            'arrival_date' => 'nullable|date',
            'description' => 'nullable|string',
        ]);

        $stockItem->update($validated);

        return redirect()->route('stock.index')
            ->with('success', 'Stock item updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $stockItem = StockItem::findOrFail($id);
        $stockItem->delete();

        return redirect()->route('stock.index')
            ->with('success', 'Stock item deleted successfully!');
    }

    /**
     * Show stock alerts
     */
    public function alerts()
    {
        $lowStockItems = StockItem::whereRaw('quantity <= alert_threshold')
            ->orderBy('quantity', 'asc')
            ->get();

        return view('stock.alerts', compact('lowStockItems'));
    }
}
