<?php

namespace App\Http\Controllers;

use App\Models\Student;
use App\Models\StockItem;
use App\Models\BookPurchase;
use App\Models\UniformPurchase;
use App\Models\LedgerEntry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SaleController extends Controller
{
    /**
     * Show book sales page
     */
    public function books()
    {
        $students = Student::orderBy('name')->get();
        $bookItems = StockItem::where('type', 'book')->orderBy('name')->get();
        $recentSales = BookPurchase::with(['student', 'stockItem'])
            ->latest('purchase_date')
            ->take(20)
            ->get();
        return view('sales.books', compact('students', 'bookItems', 'recentSales'));
    }

    /**
     * Store book sale
     */
    public function storeBook(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'stock_item_id' => 'required|exists:stock_items,id',
            'quantity' => 'required|integer|min:1',
            'unit_price' => 'required|numeric|min:0',
            'purchase_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {
            $stockItem = StockItem::findOrFail($validated['stock_item_id']);
            
            // Check if enough stock available
            if ($stockItem->quantity < $validated['quantity']) {
                throw new \Exception('Insufficient stock available. Available: ' . $stockItem->quantity);
            }

            $totalAmount = $validated['quantity'] * $validated['unit_price'];

            // Create book purchase
            $purchase = BookPurchase::create([
                'student_id' => $validated['student_id'],
                'stock_item_id' => $validated['stock_item_id'],
                'quantity' => $validated['quantity'],
                'unit_price' => $validated['unit_price'],
                'total_amount' => $totalAmount,
                'purchase_date' => $validated['purchase_date'],
                'notes' => $validated['notes'] ?? null,
            ]);

            // Reduce stock quantity
            $stockItem->decrement('quantity', $validated['quantity']);

            // Create ledger entry automatically
            LedgerEntry::create([
                'transaction_type' => 'book_purchase',
                'reference_type' => BookPurchase::class,
                'reference_id' => $purchase->id,
                'student_id' => $validated['student_id'],
                'entry_type' => 'income',
                'amount' => $totalAmount,
                'transaction_date' => $validated['purchase_date'],
                'description' => "Book Sale: {$stockItem->name} (Qty: {$validated['quantity']}) to Student #{$purchase->student->admission_number}",
            ]);
        });

        return redirect()->route('sales.books')
            ->with('success', 'Book sale recorded and ledger updated successfully!');
    }

    /**
     * Show uniform sales page
     */
    public function uniforms()
    {
        $students = Student::orderBy('name')->get();
        $uniformItems = StockItem::where('type', 'uniform')->orderBy('name')->get();
        $recentSales = UniformPurchase::with(['student', 'stockItem'])
            ->latest('purchase_date')
            ->take(20)
            ->get();
        return view('sales.uniforms', compact('students', 'uniformItems', 'recentSales'));
    }

    /**
     * Store uniform sale
     */
    public function storeUniform(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'stock_item_id' => 'required|exists:stock_items,id',
            'quantity' => 'required|integer|min:1',
            'unit_price' => 'required|numeric|min:0',
            'purchase_date' => 'required|date',
            'notes' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {
            $stockItem = StockItem::findOrFail($validated['stock_item_id']);
            
            // Check if enough stock available
            if ($stockItem->quantity < $validated['quantity']) {
                throw new \Exception('Insufficient stock available. Available: ' . $stockItem->quantity);
            }

            $totalAmount = $validated['quantity'] * $validated['unit_price'];

            // Create uniform purchase
            $purchase = UniformPurchase::create([
                'student_id' => $validated['student_id'],
                'stock_item_id' => $validated['stock_item_id'],
                'quantity' => $validated['quantity'],
                'unit_price' => $validated['unit_price'],
                'total_amount' => $totalAmount,
                'purchase_date' => $validated['purchase_date'],
                'notes' => $validated['notes'] ?? null,
            ]);

            // Reduce stock quantity
            $stockItem->decrement('quantity', $validated['quantity']);

            // Create ledger entry automatically
            LedgerEntry::create([
                'transaction_type' => 'uniform_purchase',
                'reference_type' => UniformPurchase::class,
                'reference_id' => $purchase->id,
                'student_id' => $validated['student_id'],
                'entry_type' => 'income',
                'amount' => $totalAmount,
                'transaction_date' => $validated['purchase_date'],
                'description' => "Uniform Sale: {$stockItem->name} (Qty: {$validated['quantity']}) to Student #{$purchase->student->admission_number}",
            ]);
        });

        return redirect()->route('sales.uniforms')
            ->with('success', 'Uniform sale recorded and ledger updated successfully!');
    }
}
