<?php

namespace App\Http\Controllers;

use App\Models\LedgerEntry;
use Illuminate\Http\Request;

class LedgerController extends Controller
{
    /**
     * Display ledger entries (Hesab Kitab)
     */
    public function index(Request $request)
    {
        $query = LedgerEntry::with('student')->latest('transaction_date');

        // Filter by date range
        if ($request->has('from_date')) {
            $query->where('transaction_date', '>=', $request->from_date);
        }
        if ($request->has('to_date')) {
            $query->where('transaction_date', '<=', $request->to_date);
        }

        // Filter by transaction type
        if ($request->has('transaction_type') && $request->transaction_type) {
            $query->where('transaction_type', $request->transaction_type);
        }

        // Filter by entry type
        if ($request->has('entry_type') && $request->entry_type) {
            $query->where('entry_type', $request->entry_type);
        }

        // Filter by student
        if ($request->has('student_id') && $request->student_id) {
            $query->where('student_id', $request->student_id);
        }

        $ledgerEntries = $query->paginate(20);
        
        $totalIncome = LedgerEntry::where('entry_type', 'income')
            ->when($request->from_date, fn($q) => $q->where('transaction_date', '>=', $request->from_date))
            ->when($request->to_date, fn($q) => $q->where('transaction_date', '<=', $request->to_date))
            ->sum('amount');
        
        $totalExpense = LedgerEntry::where('entry_type', 'expense')
            ->when($request->from_date, fn($q) => $q->where('transaction_date', '>=', $request->from_date))
            ->when($request->to_date, fn($q) => $q->where('transaction_date', '<=', $request->to_date))
            ->sum('amount');

        $netBalance = $totalIncome - $totalExpense;

        return view('ledger.index', compact('ledgerEntries', 'totalIncome', 'totalExpense', 'netBalance'));
    }

    /**
     * Show financial reports
     */
    public function reports(Request $request)
    {
        // Default to current month if no date range specified
        $fromDate = $request->from_date ?? now()->startOfMonth()->format('Y-m-d');
        $toDate = $request->to_date ?? now()->endOfMonth()->format('Y-m-d');

        // Overall Summary
        $totalIncome = LedgerEntry::where('entry_type', 'income')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->sum('amount');
        
        $totalExpense = LedgerEntry::where('entry_type', 'expense')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->sum('amount');

        $netBalance = $totalIncome - $totalExpense;

        // Income by Transaction Type
        $incomeByType = LedgerEntry::where('entry_type', 'income')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->selectRaw('transaction_type, SUM(amount) as total')
            ->groupBy('transaction_type')
            ->get()
            ->pluck('total', 'transaction_type');

        // Expense by Transaction Type
        $expenseByType = LedgerEntry::where('entry_type', 'expense')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->selectRaw('transaction_type, SUM(amount) as total')
            ->groupBy('transaction_type')
            ->get()
            ->pluck('total', 'transaction_type');

        // Monthly Breakdown (Last 12 months)
        $monthlyData = LedgerEntry::selectRaw('
                DATE_FORMAT(transaction_date, "%Y-%m") as month,
                SUM(CASE WHEN entry_type = "income" THEN amount ELSE 0 END) as income,
                SUM(CASE WHEN entry_type = "expense" THEN amount ELSE 0 END) as expense
            ')
            ->whereBetween('transaction_date', [now()->subMonths(11)->startOfMonth()->format('Y-m-d'), $toDate])
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        // Fee Collection Summary
        $feeCollection = LedgerEntry::where('transaction_type', 'fee_payment')
            ->where('entry_type', 'income')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->sum('amount');

        // Book Sales Summary
        $bookSales = LedgerEntry::where('transaction_type', 'book_purchase')
            ->where('entry_type', 'income')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->sum('amount');

        // Uniform Sales Summary
        $uniformSales = LedgerEntry::where('transaction_type', 'uniform_purchase')
            ->where('entry_type', 'income')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->sum('amount');

        // Top Students by Payment
        $topStudents = LedgerEntry::where('entry_type', 'income')
            ->whereBetween('transaction_date', [$fromDate, $toDate])
            ->with('student')
            ->selectRaw('student_id, SUM(amount) as total')
            ->groupBy('student_id')
            ->orderByDesc('total')
            ->take(10)
            ->get();

        // Daily Transaction Count
        $dailyTransactions = LedgerEntry::whereBetween('transaction_date', [$fromDate, $toDate])
            ->selectRaw('DATE(transaction_date) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return view('ledger.reports', compact(
            'fromDate',
            'toDate',
            'totalIncome',
            'totalExpense',
            'netBalance',
            'incomeByType',
            'expenseByType',
            'monthlyData',
            'feeCollection',
            'bookSales',
            'uniformSales',
            'topStudents',
            'dailyTransactions'
        ));
    }
}
